#!/bin/env ruby
# encoding: utf-8
require 'rubygems'
require 'rspec'
require 'watir-webdriver'
require 'page-object'
require 'oci8'
require 'watir-scroll'

require_relative '../../module/DriverUtility'
require_relative '../rspec_helper'
require_relative 'request_helper'
require_relative '../../pages/landing'
require_relative '../../pages/login'
require_relative '../../pages/eula'
require_relative '../../common/common'
require_relative '../../pages/landing_appointments'
require_relative '../../pages/appointments/new_appointment_request'
require_relative '../../pages/appointments/request_appointment_detail'
require_relative '../../helper/modal'
require_relative '../../module/database/OracleUtility'
require_relative '../../module/DateUtility'
require_relative '../../helper/error_validation'
require_relative '../../module/database/MongoUtility'


describe '[Story #VAR-1919: Appointment Request - Cancel Request]' do
  include OracleUtility
  include DateUtility
  include DriverUtility
  include MongoUtility

  before(:all) do
    initializeConfigurations(VAR_BASE_URL)
    @landing_appointments = LandingAppointments.new(@driver)
    @login = Login.new(@driver)
    @eula = Eula.new(@driver)
    @common = Common.new(@driver)
    @modal = Modal.new(@driver)
    @new_request = NewAppointmentRequest.new(@driver)
    @detail = RequestAppointmentDetail.new(@driver)
    @error = Error_Validation.new(@driver)
    TITLE = "Appointments/Requests"

    #mongo
    @db = connectToMongoDB("var-utility")
    restoreCollection("siteSupportingVAR", "var-utility", "sites-supporting-var-exported2.json", @db)
    restoreCollection("clinical-services", "var-utility", "clinical_services_default.json", @db)
    restoreCollection("custom-friendly-text", "var-utility", "facility_friendly_name.json", @db)
    restoreCollection("requestEligibilityCriteria", "var-utility", "requestEligibilityCriteria.json", @db)
    restoreCollection("directBookingEligibilityCriteria", "var-utility", "directSchedulingEnabled.json", @db)

    @today = getDateNthDaysFromNow(0, "%m/%d/%Y")

    @fiveDaysLater = getDateNthDaysFromNow(5, "%m/%d/%Y")
    @nintyDaysLater = getDateNthDaysFromNow(90, "%m/%d/%Y")

    #this will not work in CI but the data will be reset in the jenkins
    @patientID = "1006088937V099668"
    @patientID2 = "1113138327"
    deletePatientData(@patientID)
    deletePatientData(@patientID2)
    @@locationID="523A5"
    @@locationName="BROCKTON VAMC"
    @@friendlyName ="Facility Friendly Text"
    @@new_message_50 = "01234567890123456789012345678901234567890123456789"
    @common.loginEndsOnHome(pw_redacted)
    @landing_appointments.primary_header_element.when_present(TIME_OUT_LIMIT)

  end

  after(:all) do
    @driver.close
  end

  context 'AC# VAR-2047|TC# VAR-2225: Appointment Request - Cancel Request Criteria' do
    it "should create a request" do
      @common.selectAddAppointmentButton("VA")
      @new_request.selectNewAppointmentAndSchedule(EXPRESS_CARE, "523", "523", "clerk")
      @new_request.addExpressCare
      @common.waitWhileSpinnerPresent
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)
      expect(@detail.status_element.text).to eq("Status: Submitted")
    end

    it "Requests Table Validation" do
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)

      @landing_appointments.primary_header_element.when_present(TIME_OUT_LIMIT)
      request_details = @landing_appointments.request_table_list_elements[0]
      request_details.scroll.to :top
      Watir::Wait.until {@landing_appointments.request_table_list_elements.length > 0}

      #Column 1 Label: Last Updated
      expect(@landing_appointments.getNthRequestTableHeader(1)).to eq("Last Updated")
      #Column Label: Status
      expect(@landing_appointments.getNthRequestTableHeader(2)).to eq("Status")
      #Column Label: Location;
      expect(@landing_appointments.getNthRequestTableHeader(3)).to eq("Location")
      #Column Label: /Type of Care;
      expect(@landing_appointments.getNthRequestTableHeader(4)).to eq("Type of Care")
    end


    it "When a user views a request that is in Submitted status, a Cancel Request option is available in the Request Details section of the Request page" do
      Watir::Wait.until {@landing_appointments.request_table_list_elements[0].present?}
      @landing_appointments.request_table_list_elements[0].click
      @common.waitWhileSpinnerPresent

      @detail.request_details_group_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)

      expect(@detail.cancel_btn_element.exists?).to eq(true)
      expect(@detail.cancel_btn_element.text).to eq("Cancel Request")
    end

    it "When the Cancel Request button is selected a confirmation modal is shown" do
      @detail.cancel_btn_element.click
      @modal.modalTitle_element.when_present(TIME_OUT_LIMIT)
      expect(@modal.modalTitle).to eq('Confirmation')
      expect(@modal.content_element.text).to eq("Are you sure you want to cancel this request?\nSelect Yes to cancel, No to return to the Request details page.")
      expect(@modal.buttons_elements[0].text).to eq("Yes")
      expect(@modal.buttons_elements[1].text).to eq("No")
    end
    it "No button, selecting closes the modal, maintains the Submitted status for the request, " do
      @modal.buttons_elements[1].click
      @detail.request_details_group_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)
    end
    it "Should verify Yes button exists for cancel" do
      @detail.cancel_btn_element.click
      @modal.modalTitle_element.when_present(TIME_OUT_LIMIT)
      expect(@modal.modalTitle).to eq('Confirmation')
      expect(@modal.buttons_elements[0].text).to eq("Yes")
    end

    it "Selecting Yes should cancel the request" do
      @modal.buttons_elements[0].click

      @detail.status_element.when_visible(TIME_OUT_LIMIT)
      Watir::Wait.until {@detail.status_element.text == "Status: Cancelled"}
      expect(@detail.status_element.text).to eq("Status: Cancelled")
      expect(@detail.status_desc_element.text).to eq("You have successfully cancelled this request.")
      expect(@detail.status_last_updated_element.text).to include("Updated: " + @today)
    end


    it "Should verify request cancelled in list" do
      button_element = @driver.button(:id => 'back-btn')
      @driver.execute_script("$(arguments[0]).click();",button_element)
      expect(@landing_appointments.primary_header).to eq("Appointments/Requests")
    end
  end

  context "AC# VAR-2044|TC# VAR-2226: Appointment Request - Status Bar"  do
    it "Title: Status" do

      @landing_appointments.refresh_element.when_present(TIME_OUT_LIMIT)
      @landing_appointments.refresh
      @common.waitWhileSpinnerPresent
      expect(@landing_appointments.getRequestTextForRowCol(1,2)).to eq("Cancelled")
    end

    it "verify details of canceled request" do
      @landing_appointments.request_table_list_elements[0].click
      @detail.status_element.when_present(TIME_OUT_LIMIT)
      expect(@detail.has_expected_title?).to eq(true)

      expect(@detail.status_element.text).to eq("Status: Cancelled")
      expect(@detail.status_last_updated_element.text).to include("Updated: " + @today)

      expect(@detail.status_desc_element.text).to eq("You have successfully cancelled this request.")

    end

  end

end





